/*:
 * @plugindesc Displays a HUD showing the player's current Gold and Actor 113's current bet (MP). 
 * @author Clay Davis
 * @target MZ
 * @help This plugin creates a HUD overlay displaying the player's current Gold amount
 * and the current bet amount based on the MP of Actor 113 (BlackJack Bets).
 * 
 * Switch 377 (Game of Blackjack) must be ON for the HUD to display.
 */

(() => {
    class HUD_GoldBet extends Sprite {
        initialize() {
            super.initialize();
            this.createBitmap();
            this.update();
        }

        createBitmap() {
            this.bitmap = new Bitmap(300, 100); // Width and height of the HUD
            this.bitmap.fontFace = $gameSystem.mainFontFace(); // Use default font
            this.bitmap.fontSize = 23;
            this.x = Graphics.width - 310; // Position at top right of the screen
            this.y = 10; // Position near the top of the screen
            this.opacity = 200; // Transparency level (adjust as needed)
        }

        // Draw text with a black outline
        drawOutlinedText(text, x, y, width, height, align = 'left') {
            const context = this.bitmap.context;
            context.font = `${this.bitmap.fontSize}px ${this.bitmap.fontFace}`;
            context.textAlign = align;

            // Draw the outline (black color)
            context.fillStyle = 'black';
            for (let offsetX = -2; offsetX <= 2; offsetX++) {
                for (let offsetY = -2; offsetY <= 2; offsetY++) {
                    if (offsetX !== 0 || offsetY !== 0) {
                        context.fillText(text, x + offsetX, y + offsetY + this.bitmap.fontSize);
                    }
                }
            }

            // Draw the main text (white color)
            context.fillStyle = 'white';
            context.fillText(text, x, y + this.bitmap.fontSize);
        }

        update() {
            this.bitmap.clear();

            // Check if Switch 377 is ON before displaying HUD
            if (!$gameSwitches.value(377)) {
                return; // Exit early if the switch is OFF
            }

            const goldAmount = $gameParty.gold(); // Get current Gold
            const actor = $gameActors.actor(113); // Get Actor 113 (BlackJack Bets)
            const currentBet = actor ? actor.mp : 0; // Get current MP or 0 if actor doesn't exist

            // Display the current bet on the top line and Gold amount below it with a black outline
            this.drawOutlinedText(`Current Bet: ${currentBet}G`, 0, 0, 300, 36, 'left');
            this.drawOutlinedText(`Gold: ${goldAmount}G`, 0, 36, 300, 36, 'left');
        }
    }

    // Add the HUD to the map scene
    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_createAllWindows.call(this);
        this.createGoldBetHUD();
    };

    Scene_Map.prototype.createGoldBetHUD = function() {
        this._goldBetHUD = new HUD_GoldBet();
        this.addChild(this._goldBetHUD);
    };

    // Update the HUD continuously
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function(sceneActive) {
        _Scene_Map_update.call(this, sceneActive);
        if (this._goldBetHUD) {
            this._goldBetHUD.update();
        }
    };
})();